
#include "config.h"

#include "splitter.h"

#include <algorithm>
#include <cassert>
#include <cmath>
#include <limits>

#include "alnumbers.h"
#include "opthelpers.h"


template<typename Real>
void BandSplitterR<Real>::init(Real f0norm)
{
    const Real w{f0norm * (al::numbers::pi_v<Real>*2)};
    const Real cw{std::cos(w)};
    if(cw > std::numeric_limits<float>::epsilon())
        mCoeff = (std::sin(w) - 1.0f) / cw;
    else
        mCoeff = cw * -0.5f;

    mLpZ1 = 0.0f;
    mLpZ2 = 0.0f;
    mApZ1 = 0.0f;
}

template<typename Real>
void BandSplitterR<Real>::process(const al::span<const Real> input, const al::span<Real> hpout,
    const al::span<Real> lpout)
{
    const Real ap_coeff{mCoeff};
    const Real lp_coeff{mCoeff*0.5f + 0.5f};
    Real lp_z1{mLpZ1};
    Real lp_z2{mLpZ2};
    Real ap_z1{mApZ1};
    assert(lpout.size() <= input.size());
    auto lpiter = lpout.begin();
    auto proc_sample = [ap_coeff,lp_coeff,&lp_z1,&lp_z2,&ap_z1,&lpiter](const Real in) noexcept -> Real
    {
        /* Low-pass sample processing. */
        Real d{(in - lp_z1) * lp_coeff};
        Real lp_y{lp_z1 + d};
        lp_z1 = lp_y + d;

        d = (lp_y - lp_z2) * lp_coeff;
        lp_y = lp_z2 + d;
        lp_z2 = lp_y + d;

        *(lpiter++) = lp_y;

        /* All-pass sample processing. */
        Real ap_y{in*ap_coeff + ap_z1};
        ap_z1 = in - ap_y*ap_coeff;

        /* High-pass generated from removing low-passed output. */
        return ap_y - lp_y;
    };
    std::transform(input.cbegin(), input.cend(), hpout.begin(), proc_sample);
    mLpZ1 = lp_z1;
    mLpZ2 = lp_z2;
    mApZ1 = ap_z1;
}

template<typename Real>
void BandSplitterR<Real>::processHfScale(const al::span<const Real> input,
    const al::span<Real> output, const Real hfscale)
{
    const Real ap_coeff{mCoeff};
    const Real lp_coeff{mCoeff*0.5f + 0.5f};
    Real lp_z1{mLpZ1};
    Real lp_z2{mLpZ2};
    Real ap_z1{mApZ1};
    auto proc_sample = [hfscale,ap_coeff,lp_coeff,&lp_z1,&lp_z2,&ap_z1](const Real in) noexcept -> Real
    {
        /* Low-pass sample processing. */
        Real d{(in - lp_z1) * lp_coeff};
        Real lp_y{lp_z1 + d};
        lp_z1 = lp_y + d;

        d = (lp_y - lp_z2) * lp_coeff;
        lp_y = lp_z2 + d;
        lp_z2 = lp_y + d;

        /* All-pass sample processing. */
        Real ap_y{in*ap_coeff + ap_z1};
        ap_z1 = in - ap_y*ap_coeff;

        /* High-pass generated by removing the low-passed signal, which is then
         * scaled and added back to the low-passed signal.
         */
        return (ap_y-lp_y)*hfscale + lp_y;
    };
    std::transform(input.cbegin(), input.cend(), output.begin(), proc_sample);
    mLpZ1 = lp_z1;
    mLpZ2 = lp_z2;
    mApZ1 = ap_z1;
}

template<typename Real>
void BandSplitterR<Real>::processHfScale(const al::span<Real> samples, const Real hfscale)
{
    const Real ap_coeff{mCoeff};
    const Real lp_coeff{mCoeff*0.5f + 0.5f};
    Real lp_z1{mLpZ1};
    Real lp_z2{mLpZ2};
    Real ap_z1{mApZ1};
    auto proc_sample = [hfscale,ap_coeff,lp_coeff,&lp_z1,&lp_z2,&ap_z1](const Real in) noexcept -> Real
    {
        /* Low-pass sample processing. */
        Real d{(in - lp_z1) * lp_coeff};
        Real lp_y{lp_z1 + d};
        lp_z1 = lp_y + d;

        d = (lp_y - lp_z2) * lp_coeff;
        lp_y = lp_z2 + d;
        lp_z2 = lp_y + d;

        /* All-pass sample processing. */
        Real ap_y{in*ap_coeff + ap_z1};
        ap_z1 = in - ap_y*ap_coeff;

        /* High-pass generated by removing the low-passed signal, which is then
         * scaled and added back to the low-passed signal.
         */
        return (ap_y-lp_y)*hfscale + lp_y;
    };
    std::transform(samples.begin(), samples.end(), samples.begin(), proc_sample);
    mLpZ1 = lp_z1;
    mLpZ2 = lp_z2;
    mApZ1 = ap_z1;
}

template<typename Real>
void BandSplitterR<Real>::processScale(const al::span<Real> samples, const Real hfscale, const Real lfscale)
{
    const Real ap_coeff{mCoeff};
    const Real lp_coeff{mCoeff*0.5f + 0.5f};
    Real lp_z1{mLpZ1};
    Real lp_z2{mLpZ2};
    Real ap_z1{mApZ1};
    auto proc_sample = [hfscale,lfscale,ap_coeff,lp_coeff,&lp_z1,&lp_z2,&ap_z1](const Real in) noexcept -> Real
    {
        Real d{(in - lp_z1) * lp_coeff};
        Real lp_y{lp_z1 + d};
        lp_z1 = lp_y + d;

        d = (lp_y - lp_z2) * lp_coeff;
        lp_y = lp_z2 + d;
        lp_z2 = lp_y + d;

        Real ap_y{in*ap_coeff + ap_z1};
        ap_z1 = in - ap_y*ap_coeff;

        /* Apply separate factors to the high and low frequencies. */
        return (ap_y-lp_y)*hfscale + lp_y*lfscale;
    };
    std::transform(samples.begin(), samples.end(), samples.begin(), proc_sample);
    mLpZ1 = lp_z1;
    mLpZ2 = lp_z2;
    mApZ1 = ap_z1;
}

template<typename Real>
void BandSplitterR<Real>::processAllPass(const al::span<Real> samples)
{
    const Real coeff{mCoeff};
    Real z1{mApZ1};
    auto proc_sample = [coeff,&z1](const Real in) noexcept -> Real
    {
        const Real out{in*coeff + z1};
        z1 = in - out*coeff;
        return out;
    };
    std::transform(samples.cbegin(), samples.cend(), samples.begin(), proc_sample);
    mApZ1 = z1;
}


template class BandSplitterR<float>;
template class BandSplitterR<double>;
